//
//  IMAAd.h
//  GoogleIMA3
//
//  Copyright (c) 2013 Google Inc. All rights reserved.
//
//  Represents metadata of a single ad. The user can use this metadata for
//  positioning nonlinear ads (isLinear, width, height), internal tracking
//  (adId, adTitle) or custom behavior (traffickingParameters).

#import <Foundation/Foundation.h>

#import "IMAAdPodInfo.h"
#import "IMACompanionAd.h"
#import "IMAUniversalAdID.h"

NS_ASSUME_NONNULL_BEGIN

/**
 * Data object representing a single ad.
 */
@interface IMAAd : NSObject

/**
 * The ad ID as specified in the VAST response.
 */
@property(nonatomic, copy, readonly) NSString *adId;

/**
 * The ad title from the VAST response.
 */
@property(nonatomic, copy, readonly) NSString *adTitle;

/**
 * The ad description.
 */
@property(nonatomic, copy, readonly) NSString *adDescription;

/**
 * The source ad server information included in the ad response.
 */
@property(nonatomic, copy, readonly) NSString *adSystem;

/** The companion ads specified in the VAST response when using DAI. Empty for client-side ads. */
@property(nonatomic, copy, readonly) NSArray<IMACompanionAd *> *companionAds;

/**
 * Content type of the currently selected creative. For linear creatives
 * returns the content type of the currently selected media file. Returns
 * empty string if no creative or media file is selected on this ad.
 */
@property(nonatomic, copy, readonly) NSString *contentType;

/**
 * The duration of the ad from the VAST response.
 */
@property(nonatomic, readonly) NSTimeInterval duration;

/**
 * The <a href="../Enums/IMAUiElementType.html">UI elements</a> that will be displayed during ad
 * playback.
 */

@property(nonatomic, copy, readonly) NSArray<NSNumber *> *uiElements;

/**
 * Whether or not the ad UI will be disabled for this ad.
 * :nodoc:
 */
@property(nonatomic, readonly, getter=isUiDisabled) BOOL uiDisabled;

/**
 * The width of the ad asset. For non-linear ads, this is the actual width
 * of the ad representation. For linear ads, since they scale seamlessly, we
 * currently return 0 for width.
 */
@property(nonatomic, readonly) NSInteger width;

/**
 * The height of the ad asset. For non-linear ads, this is the actual height
 * of the ad representation. For linear ads, since they scale seamlessly, we
 * currently return 0 for height.
 */
@property(nonatomic, readonly) NSInteger height;

/**
 * The width of the selected creative as specified in the VAST response.
 */
@property(nonatomic, readonly) NSInteger VASTMediaWidth;

/**
 * The height of the selected creative as specified in the VAST response.
 */
@property(nonatomic, readonly) NSInteger VASTMediaHeight;

/**
 * The bitrate of the selected creative as specified in the VAST response.
 */
@property(nonatomic, readonly) NSInteger VASTMediaBitrate;

/**
 * Specifies whether the ad is linear or non-linear.
 */
@property(nonatomic, readonly, getter=isLinear) BOOL linear;

/**
 * Specifies whether the ad is skippable.
 */
@property(nonatomic, readonly, getter=isSkippable) BOOL skippable;

/**
 * The number of seconds of playback before the ad becomes skippable. -1 is returned for non
 * skippable ads or if this is unavailable.
 */
@property(nonatomic, readonly) NSTimeInterval skipTimeOffset;

/**
 * Set of ad podding properties.
 */
@property(nonatomic, readonly) IMAAdPodInfo *adPodInfo;

/**
 * String representing custom trafficking parameters from the VAST response.
 */
@property(nonatomic, copy, readonly) NSString *traffickingParameters;

/**
 * Returns the ID of the selected creative for the ad.
 */
@property(nonatomic, copy, readonly) NSString *creativeID;

/**
 * Returns the ISCI (Industry Standard Commercial Identifier) code for an ad. This
 * is the Ad-ID of the selected creative in the VAST response.
 */
@property(nonatomic, copy, readonly) NSString *creativeAdID;

/**
 * The list of all UniversalAdIds of the selected creative for this ad. Returns an empty array if
 * no universal ad IDs are found.
 */
@property(nonatomic, copy, readonly) NSArray<IMAUniversalAdID *> *universalAdIDs;

/**
 * The UniversalAdId of the selected creative for the ad. Returns the id value or "unknown"
 * if unavailable.
 */
@property(nonatomic, copy, readonly)
    NSString *universalAdIdValue DEPRECATED_MSG_ATTRIBUTE("Use universalAdIDs instead.");

/**
 * The registry associated with cataloging the UniversalAdId of the selected creative for the ad.
 * Returns the registry value, or "unknown" if unavailable.
 */
@property(nonatomic, copy, readonly)
    NSString *universalAdIdRegistry DEPRECATED_MSG_ATTRIBUTE("Use universalAdIDs instead.");

/**
 * The advertiser name as defined by the serving party.
 */
@property(nonatomic, copy, readonly) NSString *advertiserName;

/**
 * Returns the URL associated with the survey for the given ad.
 */
@property(nonatomic, copy, readonly, nullable) NSString *surveyURL;

/**
 * Returns the first deal ID present in the wrapper chain for the current ad,
 * starting from the top.
 */
@property(nonatomic, copy, readonly) NSString *dealID;

/**
 * The IDs of the ads, starting with the first wrapper ad.
 */
@property(nonatomic, copy, readonly) NSArray<NSString *> *wrapperAdIDs;

/**
 * The IDs of the ads' creatives, starting with the first wrapper ad.
 */
@property(nonatomic, copy, readonly) NSArray<NSString *> *wrapperCreativeIDs;

/**
 * Ad systems used for wrapper ads. The ad systems returned begin with the
 * first wrapper ad and continue to each wrapper ad recursively.
 */
@property(nonatomic, copy, readonly) NSArray<NSString *> *wrapperSystems;

/**
 * :nodoc:
 */
- (instancetype)init NS_UNAVAILABLE;

@end

NS_ASSUME_NONNULL_END
